package loquebot.memory;

import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.Spawn;
import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.BotDamaged;
import cz.cuni.pogamut.MessageObjects.PlayerKilled;
import cz.cuni.pogamut.MessageObjects.WallCollision;
import cz.cuni.pogamut.MessageObjects.IncommingProjectile;

import loquebot.Main;
import loquebot.util.LoqueListener;

/**
 * Responsible for listening to the messages and managing agent sences.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueSenses
{
    /** How recently the agent collided. */
    private int collision = 0;

    /**
     * Heats sence for collisions.
     * @param location Location of the collision.
     * @param normal Normal of the direction the agent collided at.
     */
    private synchronized void heatColliding (Triple location, Triple normal)
    {
        collision = 3;
    }

    /**
     * Tells, whether the agent is colliding with map geometry.
     *
     * <p><b>Note: This method clears the collision flag upon invocation.</b>
     * This is to prevent taking more action because of one collision.</p>
     *
     * @return True, if the agent is colliding with map geometry.
     */
    public synchronized boolean isColliding ()
    {
        boolean result = (collision > 0);
        collision = 0;
        return result;
    }

    /*========================================================================*/

    /** How recently the agent took damage. */
    private int beingDamaged = 0;

    /**
     * Heats sence for taking damage.
     * @param type Type of damage.
     * @param instigator ID of player causing the damage.
     */
    private synchronized void heatBeingDamaged (String type, int instigator)
    {
        beingDamaged = (int)(main.logicFrequency * .7) + 1;
    }

    /**
     * Tells, whether the agent is being damaged.
     * @return True, if the agent is being damaged.
     */
    public synchronized boolean isBeingDamaged ()
    {
        return (beingDamaged > 0);
    }

    /*========================================================================*/

    /** How recently there were enemies nearby. */
    private int enemiesNearby = 0;

    /**
     * Heats sence for spotting nearby enemies.
     * @param enemy The enemy nearby.
     */
    private synchronized void heatEnemiesNearby (Player enemy)
    {
        enemiesNearby = (int)(main.logicFrequency * .7) + 1;
    }

    /**
     * Tells, whether the agent sees any enemies nearby.
     * @return True, if the agent sees any enemies nearby.
     */
    public synchronized boolean seeEnemiesNearby ()
    {
        return (enemiesNearby > 0);
    }

    /*========================================================================*/

    /** How recently the agent was endangered by projectile. */
    private int incomingProjectile = 0;

    /**
     * Heats sence for incoming projectiles.
     * @param origin Projectile origin.
     * @param direction Projectile direction.
     * @param time Estimated time of impact
     * @param splash Splash damage radius.
     */
    private synchronized void heatIncomingProjectile (Triple origin, Triple direction, double time, double splash)
    {
        incomingProjectile = (int)(main.logicFrequency * .7) + 1;
    }

    /**
     * Tells, whether the agent is being damaged.
     * @return True, if the agent is being damaged.
     */
    public synchronized boolean seeIncomingProjectile ()
    {
        return (incomingProjectile > 0);
    }

    /*========================================================================*/

    /**
     * Fades all sences out by one step.
     * Should be called upon each agent logic loop.
     */
    protected synchronized void fadeAllOut ()
    {
        collision--;
        beingDamaged--;
        enemiesNearby--;
        incomingProjectile--;
    }

    /*========================================================================*/

    /**
     * Kills all sences.
     */
    private synchronized void killAll ()
    {
        collision = 0;
        beingDamaged = 0;
        enemiesNearby = 0;
        incomingProjectile = 0;
    }

    /**
     * Kills sences from combat.
     */
    private synchronized void killCombat ()
    {
        beingDamaged = 0;
        enemiesNearby = 0;
        incomingProjectile = 0;
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            killAll ();
        }

        /**
         * Someone fragged somebody.
         * @param msg Message to handle.
         */
        private void msgPlayerKilled (PlayerKilled msg)
        {
            // did this agent fragged?
            if (memory.self.hasID (msg.killerID))
                killCombat ();
        }

        /*========================================================================*/

        /**
         * Agent collides with map objects.
         * @param msg Message to handle.
         */
        private void msgWallCollision (WallCollision msg)
        {
            heatColliding (msg.location, msg.normal);
        }

        /**
         * Agent gets damaged.
         * @param msg Message to handle.
         */
        private void msgBotDamaged (BotDamaged msg)
        {
            heatBeingDamaged (msg.damageType, msg.idInstigator);
        }

        /**
         * Agent spots another player.
         * @param msg Message to handle.
         */
        private void msgPlayer (Player msg)
        {
            if (memory.isEnemy (msg))
                heatEnemiesNearby (msg);
        }

        /**
         * Agent spots incomming projectile.
         * @param msg Message to handle.
         */
        private void msgIncommingProjectile (IncommingProjectile msg)
        {
            heatIncomingProjectile (msg.origin, msg.direction, msg.time, msg.damageRadius);
        }

        /*========================================================================*/

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
                case PLAYER_KILLED:
                    msgPlayerKilled ((PlayerKilled) msg);
                    return;

                case WALL_COLLISION:
                    msgWallCollision ((WallCollision) msg);
                    return;
                case BOT_DAMAGED:
                    msgBotDamaged ((BotDamaged) msg);
                    return;
                case PLAYER:
                    msgPlayer ((Player) msg);
                    return;
                case INCOMMING_PROJECTILE:
                    msgIncommingProjectile ((IncommingProjectile) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener (this, MessageType.SPAWN);
            body.addTypedRcvMsgListener (this, MessageType.PLAYER_KILLED);

            body.addTypedRcvMsgListener (this, MessageType.WALL_COLLISION);
            body.addTypedRcvMsgListener (this, MessageType.BOT_DAMAGED);
            body.addTypedRcvMsgListener (this, MessageType.PLAYER);
            body.addTypedRcvMsgListener (this, MessageType.INCOMMING_PROJECTILE);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueSenses (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create listener
        this.listener = new Listener ();
    }

    /*========================================================================*/

    /**
     * Returns a string representation of the object.
     */
    @Override public String toString ()
    {
        return "collision " + collision
            + "\n beingDamaged " + beingDamaged
            + "\n enemiesNearby " + enemiesNearby
            + "\n incomingProjectile " + incomingProjectile
            ;
    }
}